"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.webhookController = exports.createApiCredential = exports.createOrganization = void 0;
const uuid_1 = require("uuid");
const Organization_1 = __importDefault(require("../models/Organization"));
const bcryptjs_1 = __importDefault(require("bcryptjs"));
const dayjs_1 = __importDefault(require("dayjs"));
const ApiCredential_1 = __importDefault(require("../models/ApiCredential"));
const CallLogs_1 = __importDefault(require("../models/CallLogs"));
const AssistantConfiguration_1 = __importDefault(require("../models/AssistantConfiguration"));
const createOrganization = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const owner_user_id = req.user.id;
        if (!owner_user_id) {
            return res.status(401).json({ message: "Unauthorized: User ID missing" });
        }
        const { name, industry_type, organization_size, assistant_name, preferred_language, assistant_task_description, contact_phone_number, status = 'active', subscription_plan_id = null, billing_contact_email = null, vapi_api_key_encrypted = null, custom_features_flags = {}, } = req.body;
        if (!name) {
            return res.status(400).json({ message: 'organization name required' });
        }
        else if (!owner_user_id) {
            return res.status(400).json({ message: 'owner are required' });
        }
        const organization = yield Organization_1.default.create({
            organization_id: (0, uuid_1.v4)(),
            name,
            owner_user_id,
            industry_type,
            organization_size,
            assistant_name,
            preferred_language,
            assistant_task_description,
            contact_phone_number,
            status,
            subscription_plan_id,
            billing_contact_email,
            vapi_api_key_encrypted,
            custom_features_flags,
        });
        console.log("suceessfully created");
        res.status(201).json({
            message: 'Organization created successfully',
            organization: {
                id: organization.organization_id,
                name: organization.name,
                industry_type: organization.industry_type,
                organization_size: organization.organization_size,
                assistant_name: organization.assistant_name,
                preferred_language: organization.preferred_language,
                assistant_task_description: organization.assistant_task_description,
                contact_phone_number: organization.contact_phone_number,
                status: organization.status
            }
        });
    }
    catch (error) {
        console.error("Error creating Organiztion", error);
        res.status(500).json({ message: "internal server Error " });
    }
});
exports.createOrganization = createOrganization;
// api Credentials 
const createApiCredential = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { id: organization_id } = req.params;
        console.log("Organization ID:", organization_id);
        const user_id = req.user.id;
        console.log("User ID:", user_id);
        const { key_name, scopes } = req.body;
        // Validate body
        if (!key_name || !Array.isArray(scopes)) {
            return res.status(400).json({ error: 'key_name and scopes are required' });
        }
        // Fetch organization
        const org = yield Organization_1.default.findByPk(organization_id);
        if (!org)
            return res.status(404).json({ error: 'Organization not found' });
        // Check owner
        if (org.owner_user_id !== user_id) {
            return res.status(403).json({ error: 'Only the organization owner can create API credentials' });
        }
        // Generate API key
        const apiKey = (0, uuid_1.v4)().replace(/-/g, '');
        const apiKeyPrefix = apiKey.slice(0, 8);
        const apiKeyHash = yield bcryptjs_1.default.hash(apiKey, 10);
        // Set expiry 1 month from now
        const expires_at = (0, dayjs_1.default)().add(1, 'month').toDate();
        const credential = yield ApiCredential_1.default.create({
            organization_id,
            user_id,
            key_name,
            api_key_hash: apiKeyHash,
            api_key_prefix: apiKeyPrefix,
            scopes,
            status: 'active',
            expires_at,
        });
        return res.status(201).json({
            message: 'API Credential created successfully',
            api_credential_id: credential.api_credential_id,
            api_key: apiKey, // Show only once
            expires_at,
        });
    }
    catch (error) {
        console.error(error);
        return res.status(500).json({ error: 'Something went wrong' });
    }
});
exports.createApiCredential = createApiCredential;
// organization webhooks endpoint 
exports.webhookController = {
    handleWebhookEvents(req, res) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const organizationId = req.params.id;
                const payload = req.body;
                console.log(`Received webhook event: ${payload.type} for organization: ${organizationId}`);
                // Verify organization exists
                const organization = yield Organization_1.default.findByPk(organizationId);
                if (!organization) {
                    return res.status(404).json({
                        success: false,
                        message: 'Organization not found'
                    });
                }
                // Handle different webhook events
                switch (payload.type) {
                    case 'call-started':
                    case 'call.started':
                        yield handleCallStarted(organizationId, payload);
                        break;
                    case 'call-ended':
                    case 'call.ended':
                        yield handleCallCompleted(organizationId, payload);
                        break;
                    case 'call-failed':
                    case 'call.failed':
                        yield handleCallFailed(organizationId, payload);
                        break;
                    default:
                        console.log(`Unhandled webhook event type: ${payload.type}`);
                        break;
                }
                return res.status(200).json({
                    success: true,
                    message: 'Webhook processed successfully'
                });
            }
            catch (error) {
                console.error('Webhook processing error:', error);
                return res.status(500).json({
                    success: false,
                    message: 'Failed to process webhook',
                    error: error instanceof Error ? error.message : 'Unknown error'
                });
            }
        });
    }
};
// Handle call started/in-progress event
function handleCallStarted(organizationId, payload) {
    return __awaiter(this, void 0, void 0, function* () {
        var _a, _b;
        try {
            const callData = payload.data.call;
            // Find assistant configuration if available
            let assistantConfigId = null;
            if (callData.assistantId) {
                const assistantConfig = yield AssistantConfiguration_1.default.findOne({
                    where: {
                        vapi_assistant_id_external: callData.assistantId,
                        organization_id: organizationId
                    }
                });
                assistantConfigId = (assistantConfig === null || assistantConfig === void 0 ? void 0 : assistantConfig.assistant_config_id) || null;
            }
            // Create or update call log
            const existingCallLog = yield CallLogs_1.default.findOne({
                where: { vapi_call_id_external: callData.id }
            });
            if (existingCallLog) {
                // Update existing call log
                yield existingCallLog.update({
                    status: 'in-progress',
                    start_time: callData.startedAt ? new Date(callData.startedAt) : new Date(),
                    caller_number: ((_a = callData.customer) === null || _a === void 0 ? void 0 : _a.number) || null,
                    direction: callData.type === 'inboundPhoneCall' ? 'inbound' : 'outbound',
                    assistant_config_id: assistantConfigId,
                    updated_at: new Date()
                });
            }
            else {
                // Create new call log
                yield CallLogs_1.default.create({
                    organization_id: organizationId,
                    assistant_config_id: assistantConfigId,
                    vapi_call_id_external: callData.id,
                    status: 'in-progress',
                    direction: callData.type === 'inboundPhoneCall' ? 'inbound' : 'outbound',
                    caller_number: ((_b = callData.customer) === null || _b === void 0 ? void 0 : _b.number) || null,
                    start_time: callData.startedAt ? new Date(callData.startedAt) : new Date(),
                    metadata_payload: {
                        webhook_event: payload.type,
                        original_payload: payload.data,
                        processed_at: new Date().toISOString()
                    }
                });
            }
            console.log(`Call started event processed for call ID: ${callData.id}`);
        }
        catch (error) {
            console.error('Error handling call started event:', error);
            throw error;
        }
    });
}
// Handle call completed event
function handleCallCompleted(organizationId, payload) {
    return __awaiter(this, void 0, void 0, function* () {
        var _a;
        try {
            const callData = payload.data.call;
            // Find assistant configuration if available
            let assistantConfigId = null;
            if (callData.assistantId) {
                const assistantConfig = yield AssistantConfiguration_1.default.findOne({
                    where: {
                        vapi_assistant_id_external: callData.assistantId,
                        organization_id: organizationId
                    }
                });
                assistantConfigId = (assistantConfig === null || assistantConfig === void 0 ? void 0 : assistantConfig.assistant_config_id) || null;
            }
            // Calculate duration
            let durationSeconds = null;
            if (callData.startedAt && callData.endedAt) {
                const startTime = new Date(callData.startedAt);
                const endTime = new Date(callData.endedAt);
                durationSeconds = Math.floor((endTime.getTime() - startTime.getTime()) / 1000);
            }
            // Update or create call log
            const existingCallLog = yield CallLogs_1.default.findOne({
                where: { vapi_call_id_external: callData.id }
            });
            if (existingCallLog) {
                // Update existing call log
                yield existingCallLog.update({
                    status: 'completed',
                    end_time: callData.endedAt ? new Date(callData.endedAt) : new Date(),
                    duration_seconds: durationSeconds,
                    cost_usd: callData.cost || null,
                    transcript_summary: callData.summary || null,
                    recording_url_external: callData.recordingUrl || null,
                    metadata_payload: Object.assign(Object.assign({}, existingCallLog.metadata_payload), { completion_data: {
                            webhook_event: payload.type,
                            transcript: callData.transcript,
                            analysis: callData.analysis,
                            messages: callData.messages,
                            processed_at: new Date().toISOString()
                        } }),
                    updated_at: new Date()
                });
            }
            else {
                // Create new call log
                yield CallLogs_1.default.create({
                    organization_id: organizationId,
                    assistant_config_id: assistantConfigId,
                    vapi_call_id_external: callData.id,
                    status: 'completed',
                    direction: callData.type === 'inboundPhoneCall' ? 'inbound' : 'outbound',
                    caller_number: ((_a = callData.customer) === null || _a === void 0 ? void 0 : _a.number) || null,
                    start_time: callData.startedAt ? new Date(callData.startedAt) : null,
                    end_time: callData.endedAt ? new Date(callData.endedAt) : new Date(),
                    duration_seconds: durationSeconds,
                    cost_usd: callData.cost || null,
                    transcript_summary: callData.summary || null,
                    recording_url_external: callData.recordingUrl || null,
                    metadata_payload: {
                        webhook_event: payload.type,
                        original_payload: payload.data,
                        processed_at: new Date().toISOString()
                    }
                });
            }
            console.log(`Call completed event processed for call ID: ${callData.id}`);
        }
        catch (error) {
            console.error('Error handling call completed event:', error);
            throw error;
        }
    });
}
// Handle call failed event
function handleCallFailed(organizationId, payload) {
    return __awaiter(this, void 0, void 0, function* () {
        var _a, _b, _c, _d, _e;
        try {
            const callData = payload.data.call;
            // Find assistant configuration if available
            let assistantConfigId = null;
            if (callData.assistantId) {
                const assistantConfig = yield AssistantConfiguration_1.default.findOne({
                    where: {
                        vapi_assistant_id_external: callData.assistantId,
                        organization_id: organizationId
                    }
                });
                assistantConfigId = (assistantConfig === null || assistantConfig === void 0 ? void 0 : assistantConfig.assistant_config_id) || null;
            }
            // Update or create call log
            const existingCallLog = yield CallLogs_1.default.findOne({
                where: { vapi_call_id_external: callData.id }
            });
            if (existingCallLog) {
                // Update existing call log
                yield existingCallLog.update({
                    status: 'failed',
                    end_time: callData.endedAt ? new Date(callData.endedAt) : new Date(),
                    error_code: ((_a = callData.error) === null || _a === void 0 ? void 0 : _a.code) || 'unknown_error',
                    error_message: ((_b = callData.error) === null || _b === void 0 ? void 0 : _b.message) || 'Call failed without specific error message',
                    metadata_payload: Object.assign(Object.assign({}, existingCallLog.metadata_payload), { failure_data: {
                            webhook_event: payload.type,
                            error: callData.error,
                            processed_at: new Date().toISOString()
                        } }),
                    updated_at: new Date()
                });
            }
            else {
                // Create new call log
                yield CallLogs_1.default.create({
                    organization_id: organizationId,
                    assistant_config_id: assistantConfigId,
                    vapi_call_id_external: callData.id,
                    status: 'failed',
                    direction: callData.type === 'inboundPhoneCall' ? 'inbound' : 'outbound',
                    caller_number: ((_c = callData.customer) === null || _c === void 0 ? void 0 : _c.number) || null,
                    start_time: callData.startedAt ? new Date(callData.startedAt) : null,
                    end_time: callData.endedAt ? new Date(callData.endedAt) : new Date(),
                    error_code: ((_d = callData.error) === null || _d === void 0 ? void 0 : _d.code) || 'unknown_error',
                    error_message: ((_e = callData.error) === null || _e === void 0 ? void 0 : _e.message) || 'Call failed without specific error message',
                    metadata_payload: {
                        webhook_event: payload.type,
                        original_payload: payload.data,
                        processed_at: new Date().toISOString()
                    }
                });
            }
            console.log(`Call failed event processed for call ID: ${callData.id}`);
        }
        catch (error) {
            console.error('Error handling call failed event:', error);
            throw error;
        }
    });
}
