"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const dotenv_1 = __importDefault(require("dotenv"));
dotenv_1.default.config();
const express_1 = __importDefault(require("express"));
const subscription_1 = __importDefault(require("./routes/subscription"));
const auth_1 = __importDefault(require("./routes/auth"));
const organization_1 = __importDefault(require("./routes/organization"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const database_1 = __importDefault(require("../config/database"));
const cors_1 = __importDefault(require("cors"));
const body_parser_1 = __importDefault(require("body-parser"));
const calendarRoutes_1 = __importDefault(require("./routes/calendarRoutes"));
const assistantConfigRoutes_1 = __importDefault(require("./routes/assistantConfigRoutes"));
// db models 
const Organization_1 = __importDefault(require("./models/Organization"));
const User_1 = __importDefault(require("./models/User"));
const Subscription_1 = __importDefault(require("./models/Subscription"));
const ApiCredential_1 = __importDefault(require("./models/ApiCredential"));
const WebhookEndpoint_1 = __importDefault(require("./models/WebhookEndpoint"));
const CallLogs_1 = __importDefault(require("./models/CallLogs"));
const AssistantConfiguration_1 = __importDefault(require("./models/AssistantConfiguration"));
const CalendarAccount_1 = __importDefault(require("./models/CalendarAccount"));
const app = (0, express_1.default)();
// Load SSL certificate and key files
const options = {
    key: fs_1.default.readFileSync(path_1.default.join(__dirname, 'key.pem')),
    cert: fs_1.default.readFileSync(path_1.default.join(__dirname, 'cert.pem')),
    ca: fs_1.default.readFileSync(path_1.default.join(__dirname, 'chain.crt'))
};
app.use(express_1.default.json());
const corsOptions = {
    origin: `${process.env.FRONTEND_URL}`,
    methods: ["GET", "POST", "PUT", "DELETE"],
    allowedHeaders: ["Content-Type", "Authorization"],
    credentials: true,
};
app.use((0, cors_1.default)(corsOptions));
subscription_1.default.use(body_parser_1.default.json());
// Sync database
database_1.default.sync()
    .then(() => {
    console.log('Database synced');
})
    .catch((error) => {
    console.error('Database sync error:', error);
});
app.get('/', (req, res) => {
    console.log("Loaded Stripe key:", process.env.STRIPE_SECRET_KEY);
    res.send('Backend is working');
});
app.use('/api/v1/payments', subscription_1.default);
app.use('/api/v1/auth', auth_1.default);
app.use('/api/v1/organization', organization_1.default);
app.use('/api/v1/calendar', calendarRoutes_1.default);
app.use("/api/v1/assistant-configs", assistantConfigRoutes_1.default);
app.use('/images', express_1.default.static(path_1.default.join(__dirname, 'public/images')));
// user + subscription 
User_1.default.hasMany(Subscription_1.default, { foreignKey: 'userId', as: 'subscriptions' });
Subscription_1.default.belongsTo(User_1.default, { foreignKey: 'userId' });
// Organization + user 
Organization_1.default.hasMany(User_1.default, { foreignKey: 'organization_id', as: 'users' });
User_1.default.belongsTo(Organization_1.default, { foreignKey: 'organization_id', as: 'Organization' });
// Organization + subscription Plan 
Organization_1.default.belongsTo(Subscription_1.default, { foreignKey: 'subscription_plan_id', as: 'subscription' });
Subscription_1.default.hasMany(Organization_1.default, { foreignKey: 'subscription_plan_id' });
// Organization has many Webhooks
Organization_1.default.hasMany(WebhookEndpoint_1.default, {
    foreignKey: 'organization_id',
    as: 'webhookEndpoint',
});
WebhookEndpoint_1.default.belongsTo(Organization_1.default, {
    foreignKey: 'organization_id',
    as: 'organization',
});
// One Organization can have many ApiCredentials
Organization_1.default.hasMany(ApiCredential_1.default, {
    foreignKey: 'organization_id', as: 'ApiCredentials'
});
ApiCredential_1.default.belongsTo(Organization_1.default, {
    foreignKey: 'organization_id', as: 'Organization'
});
// One User can have many ApiCredentials (optional field)
User_1.default.hasMany(ApiCredential_1.default, {
    foreignKey: 'user_id', as: 'ApiCredential'
});
ApiCredential_1.default.belongsTo(User_1.default, {
    foreignKey: 'user_id', as: 'users'
});
// call logs referancing 
CallLogs_1.default.belongsTo(Organization_1.default, { foreignKey: 'organization_id', as: 'Organization' });
CallLogs_1.default.belongsTo(AssistantConfiguration_1.default, { foreignKey: 'assistant_config_id', as: 'assistantConfiguration' });
CallLogs_1.default.belongsTo(ApiCredential_1.default, { foreignKey: 'api_credential_id_used', as: 'ApiCredential' });
// assistant configration refrencing by foreign keys 
AssistantConfiguration_1.default.hasMany(CallLogs_1.default, { foreignKey: 'assistant_config_id', as: 'CallLog' });
Organization_1.default.hasMany(AssistantConfiguration_1.default, { foreignKey: 'organization_id', as: 'assistantConfigs' });
CallLogs_1.default.belongsTo(AssistantConfiguration_1.default, { foreignKey: 'assistant_config_id', as: 'Assistant-Config' });
CalendarAccount_1.default.belongsTo(User_1.default, { foreignKey: 'user_id' });
User_1.default.hasMany(CalendarAccount_1.default, { foreignKey: 'user_id' });
exports.default = app;
