"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.setupPaymentMethod = exports.updatePaymentMethod = exports.createSetupIntent = exports.handleStripeWebhook = exports.switchSubscriptionPlan = exports.getUserInvoices = exports.removeUserPaymentMethod = exports.getUserPaymentMethod = exports.getSubscriptionDetails = exports.getUserCanceledSubscriptions = exports.getUserSubscriptions = exports.cancelSubscription = exports.createSubscription = void 0;
const stripe_1 = __importDefault(require("stripe"));
const sequelize_1 = require("sequelize");
const env_1 = require("../../config/env");
const Subscription_1 = __importDefault(require("../models/Subscription"));
const User_1 = __importDefault(require("../models/User"));
const stripe = new stripe_1.default(env_1.STRIPE_SECRET_KEY !== null && env_1.STRIPE_SECRET_KEY !== void 0 ? env_1.STRIPE_SECRET_KEY : "", {
    apiVersion: "2025-02-24.acacia",
});
// Define your plans and setup fees based on the frontend configuration
const PLANS = {
    "voice-ai-monthly": {
        price_id: env_1.VOICE_AI_MOTHLY_PRICE_KEY,
    },
    "voice-ai-yearly": {
        price_id: env_1.VOICE_AI_YEARLY_PRICE_KEY,
    },
    "text-ai-monthly": {
        price_id: env_1.TEXT_AI_MOTHLY_PRICE_KEY,
    },
    "text-ai-yearly": {
        price_id: env_1.TEXT_AI_YEARLY_PRICE_KEY,
    },
};
// Helper function to convert dollars to cents
const dollarsToCents = (amountInDollars) => {
    return Math.round(amountInDollars * 100);
};
// Create a subscription with a setup fee
const createSubscription = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { email, paymentMethodId, planId, userId } = req.body;
    if (!email || !paymentMethodId || !planId || !userId) {
        return res.status(400).json({ error: "Missing required fields" });
    }
    const plan = PLANS[planId];
    if (!plan) {
        return res.status(400).json({ error: "Invalid plan ID" });
    }
    try {
        const existingSubscription = yield Subscription_1.default.findOne({ where: { userId: userId } });
        let customerId = existingSubscription === null || existingSubscription === void 0 ? void 0 : existingSubscription.stripeCustomerId;
        if (!customerId) {
            // Create a customer in Stripe
            const customer = yield stripe.customers.create({
                email,
                payment_method: paymentMethodId,
                invoice_settings: {
                    default_payment_method: paymentMethodId,
                },
            });
            customerId = customer.id;
        }
        // Create a subscription with the selected plan
        const subscription = yield stripe.subscriptions.create({
            customer: customerId,
            items: [{ price: plan.price_id }],
            expand: ["latest_invoice.payment_intent"],
        });
        // Store subscription details in the database
        const dbSubscription = yield Subscription_1.default.create({
            stripeSubscriptionId: subscription.id,
            stripeCustomerId: customerId,
            stripePaymentMethodId: paymentMethodId,
            planId: planId,
            status: subscription.status,
            currentPeriodEnd: new Date(subscription.current_period_end * 1000),
            userId: userId,
        });
        // Send the subscription and payment intent details to the client
        const latestInvoice = subscription.latest_invoice;
        const paymentIntent = latestInvoice.payment_intent;
        res.status(200).json({
            subscriptionId: subscription.id,
            clientSecret: paymentIntent.client_secret,
            status: paymentIntent.status,
            subscription: dbSubscription
        });
    }
    catch (error) {
        console.error("Error creating subscription:", error);
        res.status(500).json({ error: error.message });
    }
});
exports.createSubscription = createSubscription;
// Cancel a subscription
const cancelSubscription = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { subscriptionId } = req.body;
    if (!subscriptionId) {
        return res.status(400).json({ error: "Subscription ID is required" });
    }
    try {
        // Retrieve the subscription from Stripe first to check its status
        const subscription = yield stripe.subscriptions.retrieve(subscriptionId);
        // If the subscription is in trial, cancel it immediately
        if (subscription.status === 'trialing') {
            const deletedSubscription = yield stripe.subscriptions.cancel(subscriptionId);
            // Update the subscription status in the database
            yield Subscription_1.default.update({ status: "canceled" }, { where: { stripeSubscriptionId: subscriptionId } });
            return res.status(200).json({
                success: true,
                subscription: deletedSubscription,
                message: "Trial subscription canceled successfully"
            });
        }
        // For non-trial subscriptions, cancel at period end
        const updatedSubscription = yield stripe.subscriptions.update(subscriptionId, {
            cancel_at_period_end: true
        });
        // Update the subscription status in the database
        yield Subscription_1.default.update({ status: "canceled" }, { where: { stripeSubscriptionId: subscriptionId } });
        res.status(200).json({
            success: true,
            subscription: updatedSubscription,
            message: "Subscription will be canceled at the end of the billing period"
        });
    }
    catch (error) {
        console.error("Error canceling subscription:", error);
        res.status(500).json({ error: error.message });
    }
});
exports.cancelSubscription = cancelSubscription;
// Fetch all subscriptions for a user
const getUserSubscriptions = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { userId } = req.params;
    if (!userId) {
        return res.status(400).json({ error: "User ID is required" });
    }
    try {
        const subscriptions = yield Subscription_1.default.findAll({
            where: { userId: userId },
        });
        if (!subscriptions || subscriptions.length === 0) {
            return res.status(404).json({ error: "No subscriptions found for user" });
        }
        res.status(200).json(subscriptions);
    }
    catch (error) {
        console.error("Error retrieving subscriptions:", error);
        res.status(500).json({ error: error.message });
    }
});
exports.getUserSubscriptions = getUserSubscriptions;
// Fetch all canceled subscriptions for a user
const getUserCanceledSubscriptions = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { userId } = req.params;
    if (!userId) {
        return res.status(400).json({ error: "User ID is required" });
    }
    try {
        // Retrieve all active subscriptions for the user from the database
        const currentDate = new Date();
        const canceledSubscriptions = yield Subscription_1.default.findAll({
            where: { userId: userId, status: "canceled", currentPeriodEnd: { [sequelize_1.Op.gt]: currentDate } },
        });
        if (!canceledSubscriptions || canceledSubscriptions.length === 0) {
            return res.status(404).json({ error: "No canceled subscriptions found for user" });
        }
        res.status(200).json(canceledSubscriptions);
    }
    catch (error) {
        console.error("Error retrieving active subscriptions:", error);
        res.status(500).json({ error: error.message });
    }
});
exports.getUserCanceledSubscriptions = getUserCanceledSubscriptions;
// Retrieve subscription details
const getSubscriptionDetails = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { subscriptionId } = req.params;
    if (!subscriptionId) {
        return res.status(400).json({ error: "Subscription ID is required" });
    }
    try {
        // Retrieve subscription from the database
        const subscription = yield Subscription_1.default.findOne({
            where: { stripeSubscriptionId: subscriptionId },
            include: [User_1.default],
        });
        if (!subscription) {
            return res.status(404).json({ error: "Subscription not found" });
        }
        res.status(200).json(subscription);
    }
    catch (error) {
        console.error("Error retrieving subscription:", error);
        res.status(500).json({ error: error.message });
    }
});
exports.getSubscriptionDetails = getSubscriptionDetails;
// Get payment method for a user
const getUserPaymentMethod = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e;
    const { userId } = req.params;
    if (!userId) {
        return res.status(400).json({ error: "User ID is required" });
    }
    try {
        // Retrieve the active subscription for the user from the database
        const subscription = yield Subscription_1.default.findOne({
            where: { userId: userId, stripePaymentMethodId: { [sequelize_1.Op.ne]: '' } },
            order: [['updatedAt', 'ASC']]
        });
        if (!subscription) {
            return res.status(404).json({ error: "Active subscription not found for user" });
        }
        // Retrieve the payment method associated with the subscription
        const customer = yield stripe.customers.retrieve(subscription.stripeCustomerId);
        const paymentMethod = yield stripe.customers.retrievePaymentMethod(subscription.stripeCustomerId, 
        //@ts-ignore
        ((_a = customer === null || customer === void 0 ? void 0 : customer.invoice_settings) === null || _a === void 0 ? void 0 : _a.default_payment_method) || subscription.stripePaymentMethodId);
        // Format the payment method data
        const paymentMethodResponse = {
            type: (_b = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _b === void 0 ? void 0 : _b.brand, // Example: 'Visa'
            last4: (_c = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _c === void 0 ? void 0 : _c.last4,
            expiry: `${(_d = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _d === void 0 ? void 0 : _d.exp_month}/${(_e = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _e === void 0 ? void 0 : _e.exp_year}`,
        };
        res.status(200).json(paymentMethodResponse);
    }
    catch (error) {
        res.status(500).json({ error: error });
    }
});
exports.getUserPaymentMethod = getUserPaymentMethod;
// Remove payment method for a user
const removeUserPaymentMethod = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { userId } = req.params;
    if (!userId) {
        return res.status(400).json({ error: "User ID is required" });
    }
    try {
        const activeSubscription = yield Subscription_1.default.findOne({
            where: { userId: userId, status: 'active' },
        });
        if (activeSubscription) {
            return res.status(400).json({ error: "Cannot remove payment method with active subscriptions." });
        }
        const subscriptions = yield Subscription_1.default.findAll({
            where: { userId: userId },
        });
        if (!subscriptions || subscriptions.length === 0) {
            return res.status(404).json({ error: "No subscriptions found for user" });
        }
        // Loop through each subscription to update the stripePaymentMethodId
        for (const subscription of subscriptions) {
            // Update the subscription to remove the paymentMethodId
            yield subscription.update({
                stripePaymentMethodId: '',
            });
        }
        res.status(200).json({ message: "Payment method removed from Stripe and subscriptions" });
    }
    catch (error) {
        console.error("Error removing payment method:", error);
        res.status(500).json({ error: error.message });
    }
});
exports.removeUserPaymentMethod = removeUserPaymentMethod;
// Get invoices for a user
const getUserInvoices = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { userId } = req.params;
    if (!userId) {
        return res.status(400).json({ error: "User ID is required" });
    }
    try {
        // Retrieve the active subscription for the user from the database
        const subscription = yield Subscription_1.default.findOne({
            where: { userId: userId },
        });
        if (!subscription) {
            return res.status(404).json({ error: "Active subscription not found for user" });
        }
        // Fetch both past and upcoming invoices associated with the Stripe customer
        const invoices = yield stripe.invoices.list({
            customer: subscription.stripeCustomerId,
            limit: 100, // Increase limit to get more invoices
        });
        // Format the invoices as per the required structure
        const invoiceResponse = invoices.data
            .filter(invoice => {
            // Include all non-zero amount invoices and upcoming invoices
            return invoice.total > 0 || invoice.status === 'draft' || invoice.status === 'open';
        })
            .map((invoice) => ({
            id: invoice.id,
            date: new Date(invoice.created * 1000).toISOString().split('T')[0], // Format as 'YYYY-MM-DD'
            amount: `$${(invoice.total / 100).toFixed(2)}`, // Convert cents to dollars
            status: invoice.status,
            isUpcoming: invoice.status === 'draft' || invoice.status === 'open',
            dueDate: invoice.due_date ? new Date(invoice.due_date * 1000).toISOString().split('T')[0] : null,
            nextPaymentAttempt: invoice.next_payment_attempt ? new Date(invoice.next_payment_attempt * 1000).toISOString().split('T')[0] : null,
        }));
        // Sort invoices by date, with upcoming invoices first
        invoiceResponse.sort((a, b) => {
            if (a.isUpcoming && !b.isUpcoming)
                return -1;
            if (!a.isUpcoming && b.isUpcoming)
                return 1;
            return new Date(b.date).getTime() - new Date(a.date).getTime();
        });
        res.status(200).json(invoiceResponse);
    }
    catch (error) {
        console.error("Error retrieving invoices:", error);
        res.status(500).json({ error: error.message });
    }
});
exports.getUserInvoices = getUserInvoices;
// Switch subscription plan
const switchSubscriptionPlan = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { userId, currentSubscriptionId, newPlanId } = req.body;
    if (!userId || !currentSubscriptionId || !newPlanId) {
        return res.status(400).json({ error: "Missing required fields" });
    }
    // Ensure the new plan is valid
    const newPlan = PLANS[newPlanId];
    if (!newPlan) {
        return res.status(400).json({ error: "Invalid new plan ID" });
    }
    try {
        // Find the user's current subscription in the database
        const currentSubscription = yield Subscription_1.default.findOne({
            where: { stripeSubscriptionId: currentSubscriptionId },
        });
        if (!currentSubscription) {
            return res.status(404).json({ error: "Subscription not found" });
        }
        // Retrieve the current subscription from Stripe
        const subscription = yield stripe.subscriptions.retrieve(currentSubscriptionId);
        // Ensure you're using the correct customer ID as a string
        const customerId = typeof subscription.customer === 'string' ? subscription.customer : subscription.customer.id;
        // If the subscription is in trial, we need to handle it differently
        if (subscription.status === 'trialing' && subscription.trial_end) {
            // For trial subscriptions, create a new subscription with the same trial period
            const newSubscription = yield stripe.subscriptions.create({
                customer: customerId,
                items: [{ price: newPlan.price_id }],
                expand: ["latest_invoice.payment_intent"],
                trial_end: subscription.trial_end, // Use the same trial end as current subscription
            });
            // Cancel the current trial subscription
            yield stripe.subscriptions.cancel(currentSubscriptionId);
            // Update the subscription details in the database
            yield Subscription_1.default.update({
                stripeSubscriptionId: newSubscription.id,
                planId: newPlanId,
                status: newSubscription.status,
                currentPeriodEnd: new Date(newSubscription.current_period_end * 1000),
            }, { where: { stripeSubscriptionId: currentSubscriptionId } });
            // Get the latest invoice and payment intent
            const latestInvoice = newSubscription.latest_invoice;
            const paymentIntent = latestInvoice === null || latestInvoice === void 0 ? void 0 : latestInvoice.payment_intent;
            return res.status(200).json({
                subscriptionId: newSubscription.id,
                clientSecret: paymentIntent === null || paymentIntent === void 0 ? void 0 : paymentIntent.client_secret,
                status: newSubscription.status,
                currentPeriodEnd: new Date(newSubscription.current_period_end * 1000).toISOString(),
            });
        }
        // For non-trial subscriptions, handle upgrades and downgrades differently
        const currentPriceId = subscription.items.data[0].price.id;
        const isUpgrade = newPlan.price_id !== currentPriceId;
        if (isUpgrade) {
            // For upgrades, create a new subscription that starts after the current period
            const newSubscription = yield stripe.subscriptions.create({
                customer: customerId,
                items: [{ price: newPlan.price_id }],
                expand: ["latest_invoice.payment_intent"],
                proration_behavior: "none", // This ensures no proration charges
                trial_end: Math.floor(subscription.current_period_end), // This makes the new subscription start after current period ends
            });
            // Cancel the old subscription at period end
            yield stripe.subscriptions.update(currentSubscriptionId, {
                cancel_at_period_end: true
            });
            // Update the subscription details in the database
            yield Subscription_1.default.update({
                stripeSubscriptionId: newSubscription.id,
                planId: newPlanId,
                status: newSubscription.status,
                currentPeriodEnd: new Date(newSubscription.current_period_end * 1000),
            }, { where: { stripeSubscriptionId: currentSubscriptionId } });
            // Get the latest invoice and payment intent if they exist
            const latestInvoice = newSubscription.latest_invoice;
            const paymentIntent = latestInvoice === null || latestInvoice === void 0 ? void 0 : latestInvoice.payment_intent;
            // Prepare the response
            const response = {
                subscriptionId: newSubscription.id,
                status: newSubscription.status,
                currentPeriodEnd: new Date(subscription.current_period_end * 1000).toISOString(),
            };
            // Only include payment intent details if they exist and are not zero amount
            if (paymentIntent && paymentIntent.amount > 0) {
                response.clientSecret = paymentIntent.client_secret;
                response.paymentStatus = paymentIntent.status;
            }
            res.status(200).json(response);
        }
        else {
            // For downgrades, create a new subscription that starts after the current period
            const newSubscription = yield stripe.subscriptions.create({
                customer: customerId,
                items: [{ price: newPlan.price_id }],
                expand: ["latest_invoice.payment_intent"],
                proration_behavior: "none", // This ensures no proration charges
                trial_end: Math.floor(subscription.current_period_end), // This makes the new subscription start after current period ends
            });
            // Cancel the old subscription at period end
            yield stripe.subscriptions.update(currentSubscriptionId, {
                cancel_at_period_end: true
            });
            // Update the subscription details in the database
            yield Subscription_1.default.update({
                stripeSubscriptionId: newSubscription.id,
                planId: newPlanId,
                status: newSubscription.status,
                currentPeriodEnd: new Date(newSubscription.current_period_end * 1000),
            }, { where: { stripeSubscriptionId: currentSubscriptionId } });
            res.status(200).json({
                subscriptionId: newSubscription.id,
                status: newSubscription.status,
                currentPeriodEnd: new Date(subscription.current_period_end * 1000).toISOString(),
            });
        }
    }
    catch (error) {
        console.error("Error switching subscription plan:", error);
        res.status(500).json({ error: error.message });
    }
});
exports.switchSubscriptionPlan = switchSubscriptionPlan;
// Webhook endpoint for Stripe events
const handleStripeWebhook = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const sig = req.headers["stripe-signature"];
    const endpointSecret = "your_webhook_signing_secret"; // Replace with your webhook signing secret
    let event;
    try {
        event = stripe.webhooks.constructEvent(req.body, sig, endpointSecret);
    }
    catch (error) {
        console.error("Webhook signature verification failed:", error);
        return res.status(400).json({ error: "Invalid signature" });
    }
    // Handle specific events
    switch (event.type) {
        case "invoice.payment_succeeded":
            // Handle successful payment
            break;
        case "invoice.payment_failed":
            // Handle failed payment
            break;
        case "customer.subscription.deleted":
            // Handle subscription cancellation
            const subscription = event.data.object;
            yield Subscription_1.default.update({ status: "canceled" }, { where: { stripeSubscriptionId: subscription.id } });
            break;
        case "customer.subscription.updated":
            // Handle subscription updates
            const updatedSubscription = event.data.object;
            yield Subscription_1.default.update({
                status: updatedSubscription.status,
                currentPeriodEnd: new Date(updatedSubscription.current_period_end * 1000),
            }, { where: { stripeSubscriptionId: updatedSubscription.id } });
            break;
        default:
            console.log(`Unhandled event type: ${event.type}`);
    }
    res.status(200).json({ received: true });
});
exports.handleStripeWebhook = handleStripeWebhook;
const createSetupIntent = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { userId } = req.params;
    try {
        // Find the user's subscription to get the customer ID
        const subscription = yield Subscription_1.default.findOne({
            where: { userId: userId },
        });
        if (!subscription) {
            // If no subscription exists, create a customer
            const customer = yield stripe.customers.create({
                metadata: { userId },
            });
            const setupIntent = yield stripe.setupIntents.create({
                customer: customer.id,
                usage: 'off_session',
            });
            return res.status(200).json({
                clientSecret: setupIntent.client_secret,
                customerId: customer.id
            });
        }
        // Create a setup intent for the existing customer
        const setupIntent = yield stripe.setupIntents.create({
            customer: subscription.stripeCustomerId,
            usage: 'off_session',
        });
        res.status(200).json({
            clientSecret: setupIntent.client_secret,
            customerId: subscription.stripeCustomerId
        });
    }
    catch (error) {
        console.error("Error creating setup intent:", error);
        res.status(500).json({ error: error.message });
    }
});
exports.createSetupIntent = createSetupIntent;
// Update payment method
const updatePaymentMethod = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d;
    const { userId } = req.params;
    const { newPaymentMethodId } = req.body;
    if (!userId || !newPaymentMethodId) {
        return res.status(400).json({ error: "User ID and new payment method ID are required" });
    }
    try {
        // Find the user's subscription
        const subscription = yield Subscription_1.default.findOne({
            where: { userId: userId },
        });
        if (!subscription) {
            return res.status(404).json({ error: "No subscription found for user" });
        }
        // The payment method is already attached to the customer via the setup intent
        // Set it as the default payment method
        yield stripe.customers.update(subscription.stripeCustomerId, {
            invoice_settings: {
                default_payment_method: newPaymentMethodId,
            },
        });
        // Update all active subscriptions to use the new payment method
        const subscriptions = yield Subscription_1.default.findAll({
            where: { userId: userId },
        });
        for (const sub of subscriptions) {
            if (sub.status === 'active' || sub.status === 'trialing') {
                yield stripe.subscriptions.update(sub.stripeSubscriptionId, {
                    default_payment_method: newPaymentMethodId,
                });
            }
            // Update the payment method ID in the database
            yield sub.update({
                stripePaymentMethodId: newPaymentMethodId,
            });
        }
        // Get the updated payment method details to return to the client
        const paymentMethod = yield stripe.paymentMethods.retrieve(newPaymentMethodId);
        const paymentMethodResponse = {
            type: (_a = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _a === void 0 ? void 0 : _a.brand,
            last4: (_b = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _b === void 0 ? void 0 : _b.last4,
            expiry: `${(_c = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _c === void 0 ? void 0 : _c.exp_month}/${(_d = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _d === void 0 ? void 0 : _d.exp_year}`,
        };
        res.status(200).json({
            message: "Payment method updated successfully",
            paymentMethod: paymentMethodResponse
        });
    }
    catch (error) {
        console.error("Error updating payment method:", error);
        res.status(500).json({ error: error.message });
    }
});
exports.updatePaymentMethod = updatePaymentMethod;
const setupPaymentMethod = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d;
    const { userId } = req.params;
    const { paymentMethodId } = req.body;
    if (!userId || !paymentMethodId) {
        return res.status(400).json({ error: "User ID and payment method ID are required" });
    }
    try {
        // Find the user's subscription to get the customer ID
        const subscription = yield Subscription_1.default.findOne({
            where: { userId: userId },
        });
        let customerId;
        if (!subscription) {
            // If no subscription exists, create a customer
            const customer = yield stripe.customers.create({
                metadata: { userId },
            });
            customerId = customer.id;
        }
        else {
            customerId = subscription.stripeCustomerId;
        }
        console.log(customerId);
        // 1. Attach the payment method to the customer
        const existingPaymentMethods = yield stripe.paymentMethods.list({
            customer: customerId,
            type: 'card',
        });
        if (!existingPaymentMethods.data.some(pm => pm.id === paymentMethodId)) {
            yield stripe.paymentMethods.attach(paymentMethodId, { customer: customerId });
        }
        // 2. Set it as the default payment method
        yield stripe.customers.update(customerId, {
            invoice_settings: {
                default_payment_method: paymentMethodId,
            },
        });
        // 3. If there are any active subscriptions, update them
        if (subscription) {
            const subscriptions = yield Subscription_1.default.findAll({
                where: { userId: userId },
            });
            for (const sub of subscriptions) {
                if (sub.status === 'active' || sub.status === 'trialing') {
                    yield stripe.subscriptions.update(sub.stripeSubscriptionId, {
                        default_payment_method: paymentMethodId,
                    });
                }
                // Update the payment method ID in the database
                yield sub.update({
                    stripePaymentMethodId: paymentMethodId,
                });
            }
        }
        // 4. Get the updated payment method details to return to the client
        const paymentMethod = yield stripe.paymentMethods.retrieve(paymentMethodId);
        const paymentMethodResponse = {
            type: (_a = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _a === void 0 ? void 0 : _a.brand,
            last4: (_b = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _b === void 0 ? void 0 : _b.last4,
            expiry: `${(_c = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _c === void 0 ? void 0 : _c.exp_month}/${(_d = paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.card) === null || _d === void 0 ? void 0 : _d.exp_year}`,
        };
        res.status(200).json({
            success: true,
            message: "Payment method updated successfully",
            paymentMethod: paymentMethodResponse
        });
    }
    catch (error) {
        console.error("Error setting up payment method:", error);
        // Check for Stripe errors
        if (error.type === 'StripeCardError') {
            return res.status(400).json({ error: error.message });
        }
        res.status(500).json({ error: error.message });
    }
});
exports.setupPaymentMethod = setupPaymentMethod;
